const express = require('express');
const cors = require('cors');
const fs = require('fs');
const fsp = fs.promises;
const path = require('path');
const os = require('os');
const crypto = require('crypto');
const { spawn } = require('child_process');
const analytics = require('../src/main/analytics');
const logger = require('./logger');

let ffmpegPath = null;
try {
  const ffmpegInstaller = require('@ffmpeg-installer/ffmpeg');
  ffmpegPath = ffmpegInstaller.path;

  if (ffmpegPath && ffmpegPath.includes('app.asar')) {
    const unpackedCandidate = ffmpegPath.replace('app.asar', 'app.asar.unpacked');
    if (fs.existsSync(unpackedCandidate)) {
      ffmpegPath = unpackedCandidate;
    }
  }
} catch (error) {
  // Ignore, fall back to PATH lookup later
}

const DEFAULT_PORT = 4455;
const PORT = Number(process.env.TRANSCODER_PORT) || DEFAULT_PORT;
const SESSION_TTL_MS = Number(process.env.TRANSCODER_SESSION_TTL_MS) || 5 * 60 * 1000;
const SESSION_CLEANUP_INTERVAL_MS = Number(process.env.TRANSCODER_CLEANUP_INTERVAL_MS) || 60 * 1000;
const HLS_SEGMENT_DURATION_SECONDS = Number(process.env.TRANSCODER_HLS_SEGMENT_SECONDS) || 4;
const HLS_LIST_SIZE = Number(process.env.TRANSCODER_HLS_WINDOW) || 6;
const TRANSCODER_PRESET = process.env.TRANSCODER_X264_PRESET || 'veryfast';
const TRANSCODER_CRF = process.env.TRANSCODER_X264_CRF || '20';
const MANIFEST_TIMEOUT_MS = Number(process.env.TRANSCODER_MANIFEST_TIMEOUT_MS) || 12_000;

const app = express();
app.use(cors());
app.use(express.json());

const baseTempDir = path.join(os.tmpdir(), 'miba-video-manager-hls');
fs.mkdirSync(baseTempDir, { recursive: true });

const sessions = new Map();
let cleanupTimer = null;
let serverHandle = null;
let startPromise = null;
let stopPromise = null;

function trackTranscoderEvent(eventName, properties = {}) {
  if (!eventName) {
    return;
  }
  analytics.queueAnalyticsEvent({
    event: eventName,
    source: 'transcoder',
    properties: {
      ...properties,
      active_session_count: sessions.size
    }
  });
}

function resolveFFmpegBinary() {
  if (ffmpegPath && fs.existsSync(ffmpegPath)) {
    return ffmpegPath;
  }
  const candidates = [
    process.env.FFMPEG_PATH,
    path.join(__dirname, '..', 'node_modules', '.bin', process.platform === 'win32' ? 'ffmpeg.exe' : 'ffmpeg'),
    '/opt/homebrew/bin/ffmpeg',
    '/usr/local/bin/ffmpeg',
    '/usr/bin/ffmpeg'
  ].filter(Boolean);

  for (const candidate of candidates) {
    try {
      if (candidate && fs.existsSync(candidate)) {
        let resolved = candidate;

        if (resolved.includes('app.asar')) {
          const unpackedCandidate = resolved.replace('app.asar', 'app.asar.unpacked');
          if (fs.existsSync(unpackedCandidate)) {
            resolved = unpackedCandidate;
          }
        }

        if (process.platform !== 'win32') {
          fs.accessSync(resolved, fs.constants.X_OK);
        }
        ffmpegPath = resolved;
        return resolved;
      }
    } catch (error) {
      // continue
    }
  }

  return 'ffmpeg';
}

async function ensureDir(dirPath) {
  await fsp.mkdir(dirPath, { recursive: true });
  return dirPath;
}

async function removeDir(dirPath) {
  try {
    await fsp.rm(dirPath, { recursive: true, force: true });
  } catch (error) {
    try {
      await fsp.rmdir(dirPath, { recursive: true });
    } catch (nested) {
      // Ignore residual errors
    }
  }
}

function createSessionEntry(sourcePath) {
  const id = crypto.randomUUID();
  const tempDir = path.join(baseTempDir, id);
  const manifestPath = path.join(tempDir, 'master.m3u8');
  const ffmpegExecutable = resolveFFmpegBinary();
  const fileExtension = path.extname(sourcePath || '').toLowerCase().replace('.', '') || 'unknown';

  const ffmpegArgs = [
    '-hide_banner',
    '-loglevel', 'warning',
    '-y',
    '-i', sourcePath,
    '-c:v', 'libx264',
    '-preset', TRANSCODER_PRESET,
    '-crf', TRANSCODER_CRF,
    '-c:a', 'aac',
    '-ar', '48000',
    '-ac', '2',
    '-f', 'hls',
    '-hls_time', String(HLS_SEGMENT_DURATION_SECONDS),
    '-hls_list_size', String(HLS_LIST_SIZE),
    '-hls_flags', 'delete_segments+append_list+omit_endlist',
    '-hls_segment_filename', path.join(tempDir, 'segment_%05d.ts'),
    manifestPath
  ];

  const session = {
    id,
    sourcePath,
    tempDir,
    manifestPath,
    createdAt: Date.now(),
    lastAccessedAt: Date.now(),
    ffmpeg: null,
    closed: false,
    error: null,
    readyPromise: null
  };

  session.readyPromise = (async () => {
    await ensureDir(tempDir);

    session.ffmpeg = spawn(ffmpegExecutable, ffmpegArgs, {
      stdio: ['ignore', 'ignore', 'pipe']
    });

    let stderrBuffer = '';

    session.ffmpeg.stderr?.on('data', (chunk) => {
      stderrBuffer += chunk.toString();
    });

    session.ffmpeg.on('exit', (code) => {
      session.closed = true;
      if (code !== 0 && !session.error) {
        session.error = new Error(stderrBuffer || `FFmpeg exited with code ${code}`);
      }
    });

    session.ffmpeg.on('error', (error) => {
      session.closed = true;
      session.error = error;
    });

    const startTime = Date.now();

    while (true) {
      try {
        await fsp.access(manifestPath, fs.constants.R_OK);
        break;
      } catch (accessError) {
        // ignore and fall through to check process state
      }

      if (session.closed) {
        const error = session.error || new Error('Transcoder process exited before manifest became available.');
        if (stderrBuffer) {
          logger.error('Live transcoder stderr:', stderrBuffer);
        }
        throw error;
      }

      if (Date.now() - startTime > MANIFEST_TIMEOUT_MS) {
        throw new Error('Timed out waiting for HLS manifest.');
      }

      await new Promise(resolve => setTimeout(resolve, 200));
    }

    trackTranscoderEvent('transcoder_session_ready', {
      file_extension: fileExtension,
      ffmpeg_binary: path.basename(ffmpegExecutable || 'ffmpeg')
    });

    return {
      sessionId: id,
      manifestPath
    };
  })();

  trackTranscoderEvent('transcoder_session_created', {
    file_extension: fileExtension
  });

  return session;
}

async function ensureSession(sourcePath, force = false) {
  let existing = null;
  for (const session of sessions.values()) {
    if (session.sourcePath === sourcePath) {
      existing = session;
      break;
    }
  }

  if (existing && force) {
    trackTranscoderEvent('transcoder_session_force_recycled', {
      file_extension: path.extname(existing.sourcePath || '').toLowerCase().replace('.', '') || 'unknown'
    });
    await destroySession(existing.id);
    existing = null;
  }

  if (existing) {
    existing.lastAccessedAt = Date.now();
    trackTranscoderEvent('transcoder_session_reused', {
      file_extension: path.extname(existing.sourcePath || '').toLowerCase().replace('.', '') || 'unknown'
    });
    return existing;
  }

  const session = createSessionEntry(sourcePath);
  sessions.set(session.id, session);

  try {
    await session.readyPromise;
    session.lastAccessedAt = Date.now();
    return session;
  } catch (error) {
    trackTranscoderEvent('transcoder_session_failed', {
      file_extension: path.extname(sourcePath || '').toLowerCase().replace('.', '') || 'unknown',
      error_message: error && error.message ? error.message : String(error)
    });
    await destroySession(session.id);
    throw error;
  }
}

async function destroySession(sessionId) {
  const session = sessions.get(sessionId);
  if (!session) {
    return;
  }

  sessions.delete(sessionId);

  if (session.ffmpeg && !session.closed) {
    try {
      session.ffmpeg.kill('SIGTERM');
    } catch (error) {
      // ignore
    }
    session.closed = true;
  }

  await removeDir(session.tempDir);
  trackTranscoderEvent('transcoder_session_destroyed', {
    file_extension: path.extname(session.sourcePath || '').toLowerCase().replace('.', '') || 'unknown'
  });
}

function getSession(sessionId) {
  const session = sessions.get(sessionId);
  if (session) {
    session.lastAccessedAt = Date.now();
  }
  return session;
}

async function cleanupExpiredSessions() {
  const threshold = Date.now() - SESSION_TTL_MS;
  const promises = [];

  for (const session of sessions.values()) {
    if (session.lastAccessedAt < threshold) {
      promises.push(destroySession(session.id));
    }
  }

  if (promises.length > 0) {
    await Promise.allSettled(promises);
    trackTranscoderEvent('transcoder_sessions_cleaned', { expired_count: promises.length });
  }
}

function ensureCleanupTimer() {
  if (!cleanupTimer) {
    cleanupTimer = setInterval(cleanupExpiredSessions, SESSION_CLEANUP_INTERVAL_MS);
    cleanupTimer.unref();
  }
}

app.get('/health', (_req, res) => {
  res.json({ status: 'ok' });
});

app.post('/hls/session', async (req, res) => {
  try {
    const { sourcePath, forceNew } = req.body || {};
    if (!sourcePath || typeof sourcePath !== 'string') {
      return res.status(400).json({ error: 'sourcePath is required.' });
    }

    if (!fs.existsSync(sourcePath)) {
      return res.status(404).json({ error: 'Source video not found.' });
    }

    const session = await ensureSession(sourcePath, Boolean(forceNew));
    await session.readyPromise;

    const manifestUrl = `/hls/session/${session.id}/master.m3u8`;
    trackTranscoderEvent('transcoder_session_issued', {
      force_new: Boolean(forceNew) ? 1 : 0,
      file_extension: path.extname(sourcePath || '').toLowerCase().replace('.', '') || 'unknown'
    });
    return res.json({
      sessionId: session.id,
      manifestUrl,
      segmentBaseUrl: `/hls/session/${session.id}/`,
      sourcePath
    });
  } catch (error) {
    logger.error('Failed to establish HLS session:', error);
    trackTranscoderEvent('transcoder_session_request_failed', {
      error_message: error && error.message ? error.message : String(error)
    });
    return res.status(500).json({ error: error.message || 'Failed to start transcoding session.' });
  }
});

app.get('/hls/session/:sessionId/master.m3u8', async (req, res) => {
  const session = getSession(req.params.sessionId);
  if (!session) {
    return res.status(404).json({ error: 'Session not found.' });
  }

  try {
    await fsp.access(session.manifestPath, fs.constants.R_OK);
    res.setHeader('Content-Type', 'application/vnd.apple.mpegurl');
    res.setHeader('Cache-Control', 'no-store');
    return res.sendFile(session.manifestPath);
  } catch (error) {
    logger.error('Failed to read manifest for session', session.id, error);
    return res.status(500).json({ error: 'Failed to read manifest.' });
  }
});

app.get('/hls/session/:sessionId/:segmentName', async (req, res) => {
  const session = getSession(req.params.sessionId);
  if (!session) {
    return res.status(404).json({ error: 'Session not found.' });
  }

  const segmentPath = path.join(session.tempDir, req.params.segmentName);

  try {
    await fsp.access(segmentPath, fs.constants.R_OK);
  } catch (error) {
    return res.status(404).json({ error: 'Segment not found.' });
  }

  res.setHeader('Content-Type', 'video/mp2t');
  res.setHeader('Cache-Control', 'no-store');
  return res.sendFile(segmentPath);
});

app.delete('/hls/session/:sessionId', async (req, res) => {
  try {
    await destroySession(req.params.sessionId);
    trackTranscoderEvent('transcoder_session_deleted', {
      session_id_present: req.params.sessionId ? 1 : 0
    });
    return res.status(204).end();
  } catch (error) {
    trackTranscoderEvent('transcoder_session_delete_failed', {
      error_message: error && error.message ? error.message : String(error)
    });
    return res.status(500).json({ error: 'Failed to stop session.' });
  }
});

function shutdown() {
  const activeSessions = Array.from(sessions.keys());
  Promise.allSettled(activeSessions.map((id) => destroySession(id))).finally(() => {
    process.exit(0);
  });
}

process.on('SIGINT', shutdown);
process.on('SIGTERM', shutdown);

async function startTranscoder(options = {}) {
  if (serverHandle) {
    return serverHandle;
  }
  if (startPromise) {
    return startPromise;
  }

  ensureCleanupTimer();
  const chosenPort = Number(options.port) || PORT || DEFAULT_PORT;

  startPromise = new Promise((resolve, reject) => {
    const server = app.listen(chosenPort, () => {
      serverHandle = server;
      startPromise = null;
      logger.log(`Live transcoder listening on http://localhost:${chosenPort}`);
      trackTranscoderEvent('transcoder_server_started', { port: chosenPort });
      resolve(serverHandle);
    });

    server.on('error', (error) => {
      startPromise = null;
      serverHandle = null;
      trackTranscoderEvent('transcoder_server_error', {
        port: chosenPort,
        error_message: error && error.message ? error.message : String(error)
      });
      reject(error);
    });
  });

  return startPromise;
}

async function stopTranscoder() {
  if (stopPromise) {
    return stopPromise;
  }

  stopPromise = (async () => {
    if (startPromise) {
      try {
        await startPromise;
      } catch (error) {
        // ignore failed start
      }
    }

    const activeSessions = Array.from(sessions.keys());
    if (activeSessions.length > 0) {
      await Promise.allSettled(activeSessions.map((id) => destroySession(id)));
    }

    if (cleanupTimer) {
      clearInterval(cleanupTimer);
      cleanupTimer = null;
    }

    if (serverHandle) {
      await new Promise((resolve) => {
        serverHandle.close(() => {
          serverHandle = null;
          resolve();
        });
      });
      trackTranscoderEvent('transcoder_server_stopped', {});
    }
  })();

  try {
    await stopPromise;
  } finally {
    stopPromise = null;
  }
}

async function shutdown() {
  try {
    await stopTranscoder();
  } finally {
    process.exit(0);
  }
}

if (require.main === module) {
  startTranscoder().catch((error) => {
    logger.error('Failed to start live transcoder:', error);
    process.exit(1);
  });

  process.on('SIGINT', shutdown);
  process.on('SIGTERM', shutdown);
}

module.exports = {
  app,
  startTranscoder,
  stopTranscoder,
  cleanupExpiredSessions
};

